
/* ************************************************************************ *
 *              Written by Alex de Kruijff                2009              *
 * ************************************************************************ *
 * This source was written with a tabstop every four characters             *
 * In vi type :set ts=4                                                     *
 * ************************************************************************ */

#include "configure.h"
#include "toolkit.h"

#include <errno.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif // HAVE_FCNTL_H
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif // HAVE_LOCALE_H
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif // HAVE_STDINT_H
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif // HAVE_STDLIB_H
#ifdef HAVE_STRING_H
#include <string.h>
#endif // HAVE_STRING_H
#ifdef HAVE_MMAP
#include <sys/mman.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif // HAVE_SYS_STAT_H
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif // HAVE_UNISTD_H

#include <new>

#ifdef HAVE_LOCALECONV
static char decimal_point = localeconv()->decimal_point[0];
#else // HAVE_LOCALECONV
static char decimal_point = '.';
#endif // HAVE_LOCALECONV

static struct ToolkitBuffer
{
	size_t size;
	char *buf1, *buf2;

	ToolkitBuffer();
	~ToolkitBuffer() throw();
} buffer;

ToolkitBuffer::ToolkitBuffer()
{
#ifdef HAVE_MMAP
	size = (size_t)sysconf(_SC_PAGESIZE);
#else
	size = 2^15;
#endif
	buf1 = new char[size << 1]; // throw bad_alloc
	buf2 = buf1 + size;
}

ToolkitBuffer::~ToolkitBuffer() throw()
{
	delete[] buf1;
}

int fcpy(int dst, int src, size_t size) throw()
{
	if (dst <= 0) return FILE_OPEN1_ERROR;
	if (src <= 0) return FILE_OPEN2_ERROR;
	size_t offset = 0, n = buffer.size;

#if defined(HAVE_MMAP) && defined(HAVE_MUNMAP)
	void *m2;
#endif // HAVE_MMAP && HAVE_MUNMAP

	do
	{
		if (offset + buffer.size > size)
			n = size - offset;
#if defined(HAVE_MMAP) && defined(HAVE_MUNMAP)
		if (USE_MMAP(size))
		{
			m2 = mmap(0, n, PROT_READ, MAP_SHARED, src, offset);
			if (m2 == MAP_FAILED)
				return FILE_READ2_ERROR;
			posix_madvise(m2, n, POSIX_MADV_WILLNEED);
			write(dst, buffer.buf1, n);
			posix_madvise(m2, n, POSIX_MADV_NORMAL);
			munmap(m2, n);
		}
		else
		{
#endif // HAVE_MMAP && HAVE_MUNMAP
			if (read(src, buffer.buf1, n) < 0)
				return FILE_READ2_ERROR;
			write(dst, buffer.buf1, n);
#if defined(HAVE_MMAP) && defined(HAVE_MUNMAP)
		}
#endif // HAVE_MMAP && HAVE_MUNMAP
		offset += n;
	}
	while(offset < size);
	return 0;
}

int fcpy(const char *dst, const char *src, struct stat &s) throw()
{
#ifdef O_SHLOCK
	int fd = open(dst, O_WRONLY | O_CREAT | O_SHLOCK, s.st_mode);
#else // O_SHLOCK
	int fd = open(dst, O_WRONLY | O_CREAT, s.st_mode);
#endif // O_SHLOCK
	if (fd < 0)
		return FILE_OPEN1_ERROR;
#ifdef O_SHLOCK
	int fs = open(src, O_RDONLY | O_SHLOCK);
#else // O_SHLOCK
	int fs = open(src, O_RDONLY);
#endif // O_SHLOCK
	if (fs < 0)
	{
		close(fd);
		return FILE_OPEN2_ERROR;
	}

	int status = fcpy(fd, fs, s.st_size);
	close(fd);
	close(fs);
	return status;
}

int fcmp(int fd1, int fd2, off_t s1, off_t s2) throw()
{
	if (fd1 <= 0) return FILE_OPEN1_ERROR;
	if (fd2 <= 0) return FILE_OPEN2_ERROR;

	if (s1 != s2) // diffent sizes means different 
		return FILE_DIFFERENT;
	else if (s1 == 0) // identical files if both file sizes are 0
		return FILE_IDENTICAL;

#if defined(HAVE_MMAP) && defined(HAVE_MUNMAP)
	void *m1, *m2;
#endif // HAVE_MMAP && HAVE_MUNMAP

	size_t offset = 0;
	ssize_t n = buffer.size;
	do
	{
		if (offset + buffer.size > s1)
			n = s1 - offset;
#if defined(HAVE_MMAP) && defined(HAVE_MUNMAP)
		if (USE_MMAP(s1))
		{
			m1 = mmap(0, n, PROT_READ, MAP_SHARED, fd1, offset);
			if (m1 == MAP_FAILED)
				return FILE_READ1_ERROR;
			m2 = mmap(0, n, PROT_READ, MAP_SHARED, fd2, offset);
			if (m2 == MAP_FAILED)
			{
				munmap(m2, n);
				return FILE_READ2_ERROR;
			}
			posix_madvise(m1, n, POSIX_MADV_WILLNEED);
			posix_madvise(m2, n, POSIX_MADV_WILLNEED);
			if (memcmp(m1, m2, n))
			{
				posix_madvise(m1, n, POSIX_MADV_NORMAL);
				posix_madvise(m2, n, POSIX_MADV_NORMAL);
				munmap(m1, n);
				munmap(m2, n);
				return FILE_DIFFERENT;
			}
			posix_madvise(m1, n, POSIX_MADV_NORMAL);
			posix_madvise(m2, n, POSIX_MADV_NORMAL);
			munmap(m1, n);
			munmap(m2, n);
		}
		else
		{
#endif // HAVE_MMAP && HAVE_MUNMAP
			if ((n = read(fd1, buffer.buf1, n)) < 0)
				return FILE_READ1_ERROR;
			if (n != read(fd2, buffer.buf2, n))
				return FILE_READ2_ERROR;
			if (memcmp(buffer.buf1, buffer.buf2, n))
				return FILE_DIFFERENT;
#if defined(HAVE_MMAP) && defined(HAVE_MUNMAP)
		}
#endif // HAVE_MMAP && HAVE_MUNMAP
		offset += n;
	}
	while(offset < s1);
	return FILE_IDENTICAL;
}

int fcmp(const char *f1, const char *f2, off_t s1, off_t s2) throw()
{
	if (s1 != s2) // diffent sizes means different 
		return FILE_DIFFERENT;
	else if (s1 == 0) // identical files if both file sizes are 0
		return FILE_IDENTICAL;

	// open files f1 and f2
#ifdef O_SHLOCK
	int fd1 = open(f2, O_RDONLY | O_SHLOCK);
#else // O_SHLOCK
	int fd1 = open(f1, O_RDONLY);
#endif // O_SHLOCK
	if (fd1 < 0)
		return FILE_OPEN1_ERROR;
#ifdef O_SHLOCK
	int fd2 = open(f2, O_RDONLY | O_SHLOCK);
#else // O_SHLOCK
	int fd2 = open(f2, O_RDONLY);
#endif // O_SHLOCK
	if (fd2 < 0)
	{
		close(fd1);
		return FILE_OPEN2_ERROR;
	}

	// check file input
	int status = fcmp(fd1, fd2, s1, s2);

	// Files are identical.
	close(fd1);
	close(fd2);
	return status;
}

char *fgetline(char *&str, size_t &size, FILE *file, int eol, char *ptr)
throw (std::bad_alloc)
{
	if (&size == 0)
		return NULL;
	if (ptr == NULL)
		ptr = str;
	int input = fgetc(file);
	if (input == EOF)
		return NULL;
//	size_t n = 0;
	char *end = str + size - 1;

#ifdef DEBUG
	if (ptr - str >= (int)size)
	{
		fprintf(stderr, "%s:%d ptr to far\n", __FILE__, __LINE__);
		exit(EXIT_FAILURE);
	}
#endif // DEBUG

	while(input != eol && input != EOF)
	{
#ifdef DEBUG
		if (ptr - str >= (int)size)
		{
			fprintf(stderr, "%s:%d ptr to far\n", __FILE__, __LINE__);
			exit(EXIT_FAILURE);
		}
#endif // DEBUG
		*ptr = input;
		if (++ptr == end)
		{
			char *tmp = new char[size << 1]; // throws bad_alloc
			memcpy(tmp, str, size);
			delete[] str;
			ptr = (str = tmp) + size - 1;
			end = ptr + size;
			size <<= 1;
		}
		input = fgetc(file);
	}
	*ptr = 0;
#ifdef DEBUG
	if (ptr - str >= (int)size)
	{
		fprintf(stderr, "%s:%d ptr to far\n", __FILE__, __LINE__);
		exit(EXIT_FAILURE);
	}
	if (strlen(str) >= size)
	{
		fprintf(stderr, "%s:%d string to large length %u max_size %u\n",
			__FILE__, __LINE__, strlen(str), size);
		exit(EXIT_FAILURE);
	}
#endif // DEBUG
	return str;
}

int digits(unsigned long number) throw()
{
	int counter = 0;
	while(number)
	{
		++counter;
		number /= 10;
	}
	return counter;
}

void fprintsize(FILE *out, unsigned long size) throw()
{
	int i = 0;
	while(size > 2048)
	{
		size >>= 10;
		++i;
	}
	switch(i)
	{
		case 0: fprintf(out, "%lu Bytes", size);		break;
		case 1: fprintf(out, "%lu KB", size);			break;
		case 2: fprintf(out, "%lu MB", size);			break;
		case 3: fprintf(out, "%lu GB", size);			break;
		case 4: fprintf(out, "%lu TB", size);			break;
		default: fprintf(out, "%lu ?B", size);
	}
}

static timeval tmpTime;

void fprinttime(FILE *out, struct timeval &after, struct timeval &before,
	int humanReadble) throw()
{
	tmpTime.tv_sec = after.tv_sec - before.tv_sec;
	tmpTime.tv_usec = after.tv_usec - before.tv_usec;
	if (tmpTime.tv_usec < 0)
		tmpTime.tv_sec--, tmpTime.tv_usec += 1000000;
	tmpTime.tv_usec /= 10000;
	if (humanReadble)
	{
		unsigned int days, hours, mins;
		days = tmpTime.tv_sec / 86400;
		tmpTime.tv_sec %= 86400;
		hours = tmpTime.tv_sec / 3600;
		tmpTime.tv_sec %= 3600;
		mins = tmpTime.tv_sec / 60;
		tmpTime.tv_sec %= 60;
  
		fprintf(out, "\t");
		if (days)
			fprintf(out, "%dd", days);
		if (hours)
			fprintf(out, "%dh", hours);
		if (mins)
			fprintf(out, "%dm", mins);
		fprintf(out, "%9jd%c%02lds", tmpTime.tv_sec,
			decimal_point, tmpTime.tv_usec);
	}
	else
		fprintf(out, "%9jd%c%02ld", (intmax_t)tmpTime.tv_sec,
			decimal_point, tmpTime.tv_usec);
}

#ifndef HAVE_STRSTR
#include <sys/cdefs.h>
#ifndef HAVE_STRING_H
// #error include file string.h required
#endif // HAVE_STRING_H
/*
char *strstr(const char *s, const char*find)
{
	char c, sc;
	size_t len;

	if ((c = *find++) != 0)
	{
		len = strlen(find);
		do
		{
			do
			{
				if ((sc = *s++) == 0)
					return (NULL);
			}
			while (sc != c);
		}
		while (strncmp(s, find, len) != 0);
		s--;
	}
	return (char *)s;
}

*/
#endif // HAVE_STRSTR

int createDirectory(const char *path, size_t pathOffset, const char *src)
{
	struct stat s;
	s.st_mode = 0755;
	mode_t oumask = umask(0);
	char *p = (char *)path, *tmp;
	int status = 0;
	size_t srcOffset = strlen(src) - strlen(path + pathOffset);
	for (; *p != 0; ++p)
	{
		if (*p != '/')
			continue;
		if ((size_t)(p - path) >= pathOffset)
		{
			tmp = (char *)src + srcOffset + (p - path) - pathOffset;
			*tmp = 0;
			stat(src, &s);
			*tmp = '/';
		}
		*p = 0;
		if (mkdir(path, s.st_mode) < 0)
		{
			if (errno != EEXIST)
			{
				*p = '/';
				umask(oumask);
				return DIRECTORY_FAILED;
			}
			else if (p[-1] != '/')
				status = DIRECTORY_EXISTED;
		}
		else
			status = DIRECTORY_CREATED;
		*p = '/';
	}
	umask(oumask);
	return status;
}

