
/* ************************************************************************ *
 *              Written by Alex de Kruijff                2009              *
 * ************************************************************************ *
 * This source was written with a tabstop every four characters             *
 * In vi type :set ts=4                                                     *
 * ************************************************************************ */

#include "configure.h"
#include "matchmatrix.h"

#ifdef HAVE_STRING_H
#include <string.h>
#endif // HAVE_STRING_H

#ifdef DEBUG
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif // HAVE_STDLIB_H
#endif // DEBUG

#include <new>

MatchMatrix::MatchMatrix(size_t number) throw (std::bad_alloc)
{
#ifdef DEBUG
	if (number <= 0)
	{
		fprintf(stderr, "%s:%u Number (%u) to low\n",
			__FILE__, __LINE__, number);
		exit(EXIT_FAILURE);
	}
#endif
	n = number;
	signed char *tmp = new signed char[n * sizeof(char **) + (n + 1) * n / 2];
	arr = (signed char **)tmp;
	arr[0] = ((signed char *)tmp) + n * sizeof(char **);
#ifdef HAVE_MEMSET
	memset(arr[0], 0, (n + 1) * n / 2);
#else // HAVE_MEMSET
	size_t m = (n + 1) * n / 2;
	for (size_t i = 0; i < m; ++i)
		arr[0][i] = 0;
#endif // HAVE_MEMSET

	equal = new size_t[n];
	fd = new int[n];
	mm = new void *[n];
#ifdef HAVE_MEMSET
	memset(equal, 0, n * sizeof(size_t));
	memset(fd, 0, n * sizeof(int));
#endif // HAVE_MEMSET
	for (size_t i = 1; i < n; ++i)
	{
		arr[i] = arr[i - 1] + n - i + 1;
#ifndef HAVE_MEMSET
		equal[i] = 0;
		df[i] = 0;
#endif // HAVE_MEMSET
		mm[i] = NULL;
	}

#ifdef DEBUG
	if (tmp + n * sizeof(char **) + (n + 1) * n / 2 <= arr[n - 1])
	{
		fprintf(stderr, "%s:%u Array out of bounds\n",
			__FILE__, __LINE__);
		exit(EXIT_FAILURE);
	}
#endif
}

MatchMatrix::~MatchMatrix() throw()
{
	signed char *tmp = (signed char *)arr;
	delete tmp;
	delete equal;
	delete fd;
	delete mm;
}

#ifdef DEBUG
int MatchMatrix::get(size_t i, size_t j) const throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	if (j < i)
	{
		fprintf(stderr, "%s:%u j (%u) must be larger or equal to i (%u)\n",
		__FILE__, __LINE__, j, i);
		exit(EXIT_FAILURE);
	}
	return arr[i][j - i];
}

size_t MatchMatrix::getEqual(size_t i) const throw() 
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	return equal[i];
}

int MatchMatrix::getFd(size_t i) const throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	return fd[i];
}

void *MatchMatrix::getMm(size_t i) const throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	return mm[i];
}

void MatchMatrix::increaseEqual(size_t i) const throw()  
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	++equal[i];
}

void MatchMatrix::set(size_t i, size_t j, int result) throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	if (j < i)
	{
		fprintf(stderr, "%s:%u j (%u) must be larger or equal to i (%u)",
			__FILE__, __LINE__, j, i);
		exit(EXIT_FAILURE);
	}
	arr[i][j - i] = result;
}

void MatchMatrix::setEqual(size_t i, size_t x) throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	equal[i] = x;
}

int MatchMatrix::setFd(size_t i, int x) throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	return fd[i] = x;
}

void MatchMatrix::setMm(size_t i, void *x) throw()
{
	if (i > n)
	{
		fprintf(stderr, "%s:%u i (%u) must be smaller then %u\n",
			__FILE__, __LINE__, i, n);
		exit(EXIT_FAILURE);
	}
	mm[i] = x;
}

#endif // DEBUG

void MatchMatrix::reset(size_t n) throw()
{
#ifdef DEBUG
	if (n > this->n)
	{
		fprintf(stderr, "%s:%u n (%u) must be smaller then %u\n",
			__FILE__, __LINE__, n, this->n);
		exit(EXIT_FAILURE);
	}
#endif // DEBUG
	for (size_t i = 0; i < n; ++i)
	{
#ifdef HAVE_MEMSET
		memset(arr[i], 0, n - i);
#else // HAVE_MEMSET
		for (int j = 0; j < n - 1; ++j)
			arr[i][j] = 0;
		equal[i] = 0;
		fd[i] = 0;
#endif // HAVE_MEMSET
		mm[i] = NULL;
	}
#ifdef HAVE_MEMSET
	memset(equal, 0, n * sizeof(size_t));
	memset(fd, 0, n * sizeof(int));
#endif // HAVE_MEMSET
}

