
/* ************************************************************************ *
 * ************************************************************************ * 
 *              Written by Alex de Kruijff                2009              *
 * ************************************************************************ * 
 * This source was written with a tabstop every four characters             * 
 * In vi type :set ts=4                                                     * 
 * ************************************************************************ */

#include "mainAction.h"

#if defined(HAVE_SYS_TIME_H) && defined(HAVE_GETTIMEOFDAY)
#include <sys/time.h>
#endif // HAVE_SYS_TIME_H && HAVE_GETTIMEOFDAY

// Retrieved from actionProcessOptions
static char *program = NULL;
static const char *sep = "\t";
static off_t minSize = 0, maxSize = UINT_MAX;
static int stopping = 0;
unsigned int flags = VERBOSE_LEVEL1 | MATCH_LEFT;

// Retrieved from actionProcessInput
static size_t processed = 0, untouched = 0, lines = 0;
static ulongest_t bytesSaved = 0;
static struct timeval time0, time1;

static void quit(int sig)
{
	stopping = 1;
}

size_t getPath(char *&path, size_t pathOffset, size_t capacity,
	const char *src, size_t srcOffset)
{
	size_t srcLen = strlen(src);
	if (pathOffset + srcLen - srcOffset >= capacity)
	{
		while(pathOffset + srcLen - srcOffset >= capacity)
			capacity <<= 1;
		char *tmp = new char[capacity];
		memcpy(tmp, path, pathOffset);
		delete[] path;
		path = tmp;
	}
	if (srcLen >= srcOffset + 2 &&
		src[srcOffset] == '.' && src[srcOffset + 1] == '/')
		++srcOffset;
	if (path[strlen(path) - 1] == '/' && src[srcOffset] == '/')
		++srcOffset; 
	memcpy(path + pathOffset, src + srcOffset, srcLen - srcOffset);
	return capacity;
}

size_t getParameter(char *&str, char *param, size_t capacity)
{
	size_t len = strlen(str), paral = strlen(param);
	if (len + paral + 1 >= capacity)
	{
		while(len + paral + 1 >= capacity)
			capacity <<= 1;
		char *tmp = new char[capacity];
		memcpy(tmp, str, len);
		tmp[len] = 0;
		delete[] str;
		str = tmp;
	}
	memcpy(str + len, param, paral + 1);
	return capacity;
}

void actionUsage(const char *command)
{
	fprintf(stderr, "\n%s reads the samefile output and links", program); 
	fprintf(stderr, "\nidentical files together.\n");
	fprintf(stderr, "\nusage: %s [-A  |  -At  |  -L  |  -Z | -Zt] [-g size] \\", program);
	size_t len = strlen(program);
	fprintf(stderr, "\n        ");
	for (size_t i = 0; i < len; ++i)
		fprintf(stderr, " ");
	fprintf(stderr, "[-m size] [-S sep] [-HnqstVvw]\n");

	fprintf(stderr, "example: find <dir> | samefile | %s\n", program);
	fprintf(stderr, "\n");
	fprintf(stderr, "\n  options: -A %s based on the first filename", command);
	fprintf(stderr, "\n           -g only process files greater than size (0)");
//	fprintf(stderr, "\n           -H human readable statistics");
	fprintf(stderr, "\n           -L %s based on the number of links (default)", command);
	fprintf(stderr, "\n           -m only process files less or equal than size (0)");
	fprintf(stderr, "\n           -n perform a trial run with no changes made");
	fprintf(stderr, "\n           -q suppress non-error messages");
	fprintf(stderr, "\n           -S use sep as separator string for files (tab)");
	fprintf(stderr, "\n           -s symlink files if its not posible to hard the files");
	fprintf(stderr, "\n           -t match based on the modiciation time instead of the path");
	fprintf(stderr, "\n           -V output version information and exit");
	fprintf(stderr, "\n           -v increase verbosity");
	fprintf(stderr, "\n           -w don't check files contence");
	fprintf(stderr, "\n           -Z %s based on the last filename", command);
	fprintf(stderr, "\n");
	exit(EXIT_SUCCESS);
}

int actionProcessOptions(int argc, char **argv, const char *command)
{
#ifdef SIGALRM
	signal(SIGALRM, quit);
#endif // SIGALRM
#ifdef SIGHUP
	signal(SIGHUP, quit);
#endif // SIGHUP
#ifdef SIGINT
	signal(SIGINT, quit);
#endif // SIGINT
#ifdef SIGPIPE
	signal(SIGPIPE, quit);
#endif // SIGPIPE
#ifdef SIGPROF
	signal(SIGPROF, quit);
#endif // SIGPROF
#ifdef SIGQUIT
	signal(SIGQUIT, quit);
#endif // SIGQUIT
#ifdef SIGTERM
	signal(SIGTERM, quit);
#endif // SIGTERM
#ifdef SIGTHR
	signal(SIGTHR, quit);
#endif // SIGTHR
#ifdef SIGUSR1
	signal(SIGUSR1, quit);
#endif // SIGUSR1
#ifdef SIGUSR2
	signal(SIGUSR2, quit);
#endif // SIGUSR2
#ifdef SIGVTALRM
	signal(SIGVTALRM, quit);
#endif // SIGVTALRM
#ifdef SIGXCPU
	signal(SIGXCPU, quit);
#endif // SIGXCPU
#ifdef SIGXFSZ
	signal(SIGXFSZ, quit);
#endif // SIGVTALRM
	(program = rindex(argv[0], '/')) ? ++program : program = argv[0];

	int c;
	while ((c = getopt(argc, argv, "h?g:S:aHALZtisnqVvw")) != -1)
		switch(c)
		{
			default: case 'h': case '?': actionUsage(command);				break;

			case 'g':
				if (sscanf(optarg, "%ju", &minSize) != 1)
					minSize = 0, fprintf(stderr,
						"warning: can't convert -g %s, using -g 0 instead\n",
						optarg);
				break;
			case 'm':
				if (sscanf(optarg, "%ju", &maxSize) !=  1)
					maxSize = 0, fprintf(stderr,
						"warning: can't convert -m %s, using-g 0 instead\n",
						optarg);
				break;

			case 'S': sep = optarg;										break;
			case 'H': flags |= HUMAN_READABLE;							break;

			case 'A': flags |=  MATCH_LEFT;		flags &= ~MATCH_RIGHT;	break;
			case 'L': flags |=  MATCH_LEFT;		flags |=  MATCH_RIGHT;  break;
			case 'Z': flags &= ~MATCH_LEFT;		flags |=  MATCH_RIGHT;	break;
			case 't': flags |= MATCH_TIME;								break;

			case 's': flags |= SYMLINK;									break;
			case 'n': flags |= DRYRUN;									break;

			case 'q': flags &= ~VERBOSE_MASK;							break;
			case 'V':
				printf(COPYRIGHT, PACKAGE_STRING, program);
				exit(EXIT_SUCCESS);
			break;
			case 'v': if ((S_VERBOSE(flags)) < VERBOSE_MAX) ++flags;	break;

			case 'w': flags &= WITHOUT_FILE_CHECK;						break;
		}
	return optind;
}

void actionProcessInput(int (&func)(const char *, const char *,
	const struct stat &, const struct stat &, 
	const char *, const char *, char *))
{
#ifdef HAVE_GETTIMEOFDAY
	gettimeofday(&time0, (struct timezone *)NULL);
#endif // HAVE_GETTIMEOFDAY
	int len = strlen(sep), result;
	size_t fs = 1024, capacity = 1024;
	char *line = new char[capacity];

	size_t size = 0;
	match_t m;
	Cache cache;
	char *f1 = new char[fs], *f2 = new char[fs];
	char *backup = new char[fs];
	while(fgetline(line, capacity, stdin) != NULL)
	{
		++lines;
		if (stopping)
			break;

		// Are f1 and f2 to small?
		if (strlen(line) > fs + 21)
		{
			delete[] f1;
			delete[] f2;
			delete[] backup;
			while(strlen(line) > fs)
				fs <<= 1;
			f1 = new char[fs];
			f2 = new char[fs];
			backup = new char[fs];
		}

		// read input
		if (inputSamefile(line, size, f1, f2, sep, len))
		{
			printf("%s\n", line);
			++untouched;
			continue;
		}

		// get meta data
		if (lstat(f1, &m.s1) < 0 || lstat(f2, &m.s2))
		{
			printf("%s\n", line);
			++untouched;
			continue;
		}

		// check the file size
		if (m.s1.st_size <= minSize || m.s2.st_size <= minSize ||
			m.s1.st_size > maxSize || m.s2.st_size > maxSize ||
			!S_SYMLINK(flags) && m.s1.st_dev != m.s2.st_dev)
		{
			outputSamefile(f1, f2, m.s1.st_nlink, m.s2.st_nlink, m.s1.st_size, 
				m.s1.st_dev == m.s2.st_dev, sep);
			++untouched;
			continue;
		}

		// compare if files are realy equal.
		if (!S_WITHOUT_FILE_CHECK(flags))
		{
                        if (cache != m) // check f1 with f2
                        {
                                m.result = fcmp(f1, f2, m.s1.st_size, m.s2.st_size);
                                cache += m;
                        }
			switch(cache[m].result)
			{
				case FILE_OPEN1_ERROR:
				case FILE_OPEN2_ERROR:
				case FILE_READ1_ERROR:
				case FILE_READ2_ERROR:
				case FILE_UNKOWN:
				case FILE_DIFFERENT:
				default:
					++untouched;
					continue;
				case FILE_IDENTICAL:
					break;
			}
		}

		const char *dst, *src;
		switch(S_MATCH(flags))
		{
			default:
			case MATCH_LEFT:					// -A
				if (strcmp(f1, f2) < 0)			dst = f1, src = f2;
				else							dst = f2, src = f1;
				break;
			case MATCH_LEFT | MATCH_TIME:		// -At
				if (m.s1.st_mtime < m.s2.st_mtime)	dst = f1, src = f2;
				else							dst = f2, src = f1;
				break;
			case MATCH_RIGHT:					// -Z
				if (strcmp(f1, f2) < 0)			dst = f2, src = f1;
				else							dst = f1, src = f2;
				break;
			case MATCH_RIGHT | MATCH_TIME:		// -Zt
				if (m.s1.st_mtime < m.s2.st_mtime)	dst = f2, src = f1;
				else							dst = f1, src = f2;
				break;
			case MATCH_LEFT | MATCH_RIGHT:		// -L
				if (m.s1.st_nlink <= m.s2.st_nlink)	dst = f2, src = f1;
				else							dst = f1, src = f2;
				break;
		}

		switch(result = func(f1, f2, m.s1, m.s2, dst, src, backup))
		{
			case SUCCES_LEFT:
				if (m.s1.st_nlink == 1)
					bytesSaved += m.s1.st_size;
				++processed;
			break;

			case SUCCES_RIGHT:
				if (m.s2.st_nlink == 1)
					bytesSaved += m.s2.st_size;
				++processed;
			break;

			case DISMISS_SILENTLY:
				++untouched;
			break;

			default:
				++untouched;
				outputSamefile(f1, f2, m.s1.st_nlink, m.s2.st_nlink, m.s1.st_size, 
					m.s1.st_dev == m.s2.st_dev, sep);
		}

		if (S_VERBOSE_LEVEL1(flags))
			switch(result)
			{
				case FAILED_BACKUP_CREATE:
					fprintf(stderr, "failed to create a backup %s\n",  f1);
				break;

				case FAILED_BACKUP_DELETE:
					fprintf(stderr, "failed to delete a backup %s\n",  f1);
				break;

				case FAILED_REMOVE_LEFT:
					fprintf(stderr, "failed to remove %s\n",  f1);
				break;

				case FAILED_REMOVE_RIGHT:
					fprintf(stderr, "failed to remove %s\n",  f2);
				break;

				case FAILED_LINK_LEFT:
					fprintf(stderr, "failed to link %s -> %s\n",  f1, f2);
				break;

				case FAILED_LINK_RIGHT:
					fprintf(stderr, "failed to link %s -> %s\n",  f2, f1);
				break;
			}
	}
	delete[] line;
	delete[] f1;
	delete[] f2;
	delete[] backup;
#ifdef HAVE_GETTIMEOFDAY
	gettimeofday(&time1, (struct timezone *)NULL);
#endif // HAVE_GETTIMEOFDAY
}

void actionProcessStats()
{
	fprintf(stderr, "\nProcessed files: %u", (unsigned)processed);
	fprintf(stderr, "\nUntouched files: %u", (unsigned)untouched);
#ifdef __LONG_LONG_SUPPORTED
	fprintf(stderr, "\nBytes saved....: %llu", bytesSaved);
#else // __LONG_LONG_SUPPORTED
	fprintf(stderr, "\nBytes saved....: %lu", bytesSaved);
#endif // __LONG_LONG_SUPPORTED
	fprintf(stderr, "\nExecution time:");
#ifdef HAVE_GETTIMEOFDAY
	fprinttime(stderr, time1, time0, S_HUMAN_READABLE(flags));
#endif // HAVE_GETTIMEOFDAY
	fprintf(stderr, "\n");
}

