
/* ************************************************************************ *
 *              Written by Alex de Kruijff                2009              *
 * ************************************************************************ *
 * This source was written with a tabstop every four characters             *
 * In vi type :set ts=4                                                     *
 * ************************************************************************ */

#include "configure.h"
#include "hash.h"
#include "toolkit.h"
#include "matchmatrix.h"
#include "sizegroup.h"
#include "holder.h"
#include "visitor.h"
#include "stats.h"

#include <new>

/* ************************************************************************ */
SizeGroup Holder::tmp;

#ifndef NO_EXPERIMENTAL
#define EXPERIMENTAL
#endif // NO_EXPERIMENTAL

Holder::Holder(size_t capacity) throw (std::bad_alloc) : hash(capacity)
{
	hash.setHashFunction(SizeGroup::hashFunction);
}

size_t Holder::remove(off_t min, off_t max) throw()
{
	size_t counter = 0, n = hash.getBoundry();
	for (size_t i = 0; i < n; ++i) if (hash[i] != NULL)
		if (min <= hash[i]->getFileSize() && hash[i]->getFileSize() < max)
		{
			hash.deleteItem(i);
			++counter;
		}
	hash.fix();
	return counter;
}

SizeGroup &Holder::operator[](const struct stat &s) throw (std::bad_alloc)
{
	tmp = s;
	if (hash[tmp] != NULL)
		return *hash[tmp];

	SizeGroup *ptr = new SizeGroup(s); // throws bad_alloc
	try
	{
		hash += *ptr; // throws bad_alloc
		return *ptr;
	}
	catch(std::bad_alloc &e)
	{
		delete ptr;
		throw(e);
	}
}

void Holder::sort(int (&compare)(const void *a, const void *b),
	int (&compareFilename)(const void *a, const void *b)) throw()
{
	hash.sort(SizeGroup::compare); // converts the container to an array
	size_t n = hash.getSize();
	for (size_t i = 0; i < n; ++i)
		hash[i]->sort(compare, compareFilename);
}

void Holder::accept(SamefileVisitor &v)
{
	if (v.visit(*this))
		return;
	size_t n = hash.getBoundry();
	for (size_t i = 0; i < n; ++i)
		if (hash[i] != NULL)
			hash[i]->accept(v);
}

ulongest_t Holder::compareFiles(Stats &stats,
	int (&func)(const SizeGroup &, const FileGroup &, 
		const Filename &, const FileGroup &, const Filename &, int),
	int flags,
	int (&addingAllowed)(const char *, const FileGroup &),
	int (*postAction)(SizeGroup &, size_t, size_t),
	int (*preCheck)(const SizeGroup &, const FileGroup &, const FileGroup &)
	) throw(std::bad_alloc)
{
	// Clean up SizeGroup(s) with just one FileGroup 
	// And get the maximum FileGroup(s) within any one SizeGroup
	size_t max = 1, size, n = hash.getBoundry();
	for (size_t i = 0; i < n; ++i)
	{
		if (hash[i] == NULL)
			continue;
		if ((size = hash[i]->getSize()) > max)
			max = size;
	}

	// Compare
	ulongest_t waisted;
	MatchMatrix match(max); // throws(bad_alloc)
#ifdef DEBUG
	fprintf(stderr, "debug: largest SizeGroup contains %i files\n", max);
#endif // DEBUG
	for (size_t i = 0; i < n; ++i)
	{
		if (hash[i] == NULL)
			continue;
		SizeGroup &select = *hash[i];
#ifdef WITH_DISK_STORAGE
		select.diskRead(addingAllowed);
#endif // WITH_DISK_STORAGE
		waisted += select.getFileSize() *
			select.compareFiles(match, func, flags, preCheck);
		hash[i]->accept(stats);
		if (postAction != NULL && postAction(select, i, n))
			hash.deleteItem(i);
	}
	return waisted;
}

